/**
 * \file: AlsaAudioCommon.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <memory.h>
#include <adit_logging.h>
#include "AlsaAudioCommon.h"

using namespace std;

LOG_IMPORT_CONTEXT(aauto_audio)

#define SILENCE_PREFILL_DEFAULT 2
#define BUFFER_PERIODS_DEFAULT 2
//#define getErrAsStr(X) (X == EINVAL) ? "EINVAL" : (X == EPERM) ? "EPERM" : "UNKNOWNERR"

namespace adit { namespace aauto
{

AlsaAudioCommon::AlsaAudioCommon(bool inDumpConfiguration, bool inVerboseLogging)
{
    dump_config = inDumpConfiguration;
    verbose = inVerboseLogging;

    pcm = NULL;
    silence_buffer = NULL;

    /* initial settings */
    dir = SND_PCM_STREAM_PLAYBACK;
    format = SND_PCM_FORMAT_S16_LE;
    channels = 2;
    rate = 48000;
    ideal_period_ms = 8;
    buffer_ideal_periods = -1;
    prefill_ms = -1;
    bytes_per_frame = 16;
    buffer_size = 1152;
    period_size = 0;
    /* PCM access type will be set in SetupPCM() */
    access = SND_PCM_ACCESS_MMAP_INTERLEAVED;
    ideal_period_size = 384;
    silence_prefill = 0;

    recover_delay_ms = 0;

    streamName = NULL;

    /* set default to 0. No pre-buffering */
    threshold_ms = 0;
    threshold_frames = 0;
    threshold_buffer_size = 0;

    /* set default timeout */
    init_tout = (int)(ideal_period_ms + 20);
    startup_xfer = 0;

    setAlsaAudioState(AlsaAudioStop);
    mSetupPCM = false;
    mStreamStarted = false;
    mStreamStopped = false;
    mStreamBroken = false;
    wait_done = false;
}

AlsaAudioCommon::~AlsaAudioCommon()
{
    ReleasePCM();
}

#define DumpHwSwParams(func,stream,attoutput) \
        if (attoutput != NULL) {\
            snd_output_printf(attoutput, "Dumping HwParams for %s at %s", stream, func);\
            snd_pcm_hw_params_dump(hw_params, attoutput);\
            snd_output_printf(attoutput, "Dumping SwParams for %s at %s", stream, func);\
            snd_pcm_sw_params_dump(sw_params, attoutput);\
            snd_output_flush (attoutput);\
        }

#define LogSndStdErrorAndReturn(err,func,stream,attoutput) \
    if (err < 0) {\
        LOG_ERROR((aauto_audio, "%s failed with err %d (%s) for stream %s",\
            func, err, snd_strerror(err), stream));\
        DumpHwSwParams(func, stream, attoutput);\
        if (attoutput != NULL)\
            snd_output_close(attoutput);\
        return err;\
    }

void AlsaAudioCommon::SetStreamName(const char *audio_stream_name)
{
    streamName = audio_stream_name;
}

void AlsaAudioCommon::SetDir(snd_pcm_stream_t dir_)
{
    dir = dir_;
}
void AlsaAudioCommon::SetFormat(snd_pcm_format_t format_)
{
    format = format_;
}
void AlsaAudioCommon::SetChannels(unsigned int channels_)
{
    channels = channels_;
}
void AlsaAudioCommon::SetRate(unsigned int rate_)
{
    rate = rate_;
}
void AlsaAudioCommon::SetInitTout(int tout_)
{
    init_tout = tout_;
}
void AlsaAudioCommon::SetThresholdMs(unsigned int threshold_ms_)
{
    threshold_ms = threshold_ms_;
}
void AlsaAudioCommon::SetIdealPeriodMs(unsigned int ideal_period_ms_)
{
    if (ideal_period_ms_ > 0) {
        ideal_period_ms = ideal_period_ms_;
    } else {
        LOG_INFO((aauto_audio, "%s, Time for ideal period has to be >%u. Use default %u",
                  streamName, ideal_period_ms_, ideal_period_ms));
    }
}
void AlsaAudioCommon::SetBufferIdealPeriods (int buffer_ideal_periods_)
{
    if (buffer_ideal_periods_ >= 0) {
        if (buffer_ideal_periods_ < 2) {
            LOG_ERROR((aauto_audio, "%s, Buffer ideal periods has to be >= 2", streamName));
            return;
        }
        buffer_ideal_periods = buffer_ideal_periods_;
    } else {
        buffer_ideal_periods = -1;
    }
}
void AlsaAudioCommon::SetPrefillMs(int prefill_ms_)
{
    if (prefill_ms_ >= 0) {
        prefill_ms = prefill_ms_;
    } else {
        prefill_ms = -1;
    }
}
void AlsaAudioCommon::SetRecoverDelayMs(unsigned int recover_delay_ms_)
{
    recover_delay_ms = recover_delay_ms_;
}

snd_pcm_uframes_t AlsaAudioCommon::GetIdealPeriodMs()
{
    if (mSetupPCM == false)
        return 0;
    return ideal_period_size;
}

snd_pcm_uframes_t AlsaAudioCommon::GetPrefillMs()
{
    if (mSetupPCM == false)
        return 0;
    return silence_prefill;
}

size_t AlsaAudioCommon::GetIdealPeriodBytes()
{
    if (mSetupPCM == false)
        return 0;
    return (size_t)(ideal_period_size * bytes_per_frame);
}

int AlsaAudioCommon::GetThresholdBufferSize(unsigned int samples_per_frame_)
{
    if (mSetupPCM == false)
        return 0;

    /* Calculate the Jitter/playtime - expected time between two dataAvailableCallbacks from MD */
    unsigned int jitterMs = samples_per_frame_ / (rate / 1000);

    /* Calculate the (ALSA) frame size for the threshold buffer */
    threshold_frames = ( threshold_ms + jitterMs) * (ideal_period_size / ideal_period_ms);

    /* Calculate the threshold buffer size in bytes */
    threshold_buffer_size = threshold_frames * bytes_per_frame;

    if (verbose) {
        // TODO: debug or verbose log level?
        LOGD_DEBUG((aauto_audio, "%s, jitter=%d ms, threshold_ms=%d, threshold_frames=%d, threshold_buffer_size=%d",
                streamName, jitterMs, threshold_ms, threshold_frames, threshold_buffer_size));
    }

    return threshold_buffer_size;
}

int AlsaAudioCommon::SetupPCM(const char *pcmname)
{
    int err;
    snd_pcm_hw_params_t *hw_params;
    snd_pcm_sw_params_t *sw_params;
    unsigned int i;
    unsigned int chmin, chmax;
    int prefill_ms_local;
    int buffer_periods_local;
    const unsigned char dividers[] =
    { 2, 3, 5, 6, 7, 10, 0 }; /*zero terminated list of dividers*/
    unsigned int mydiv;
    snd_output_t *output = NULL;

    /*setup access types in preferred order*/
    const snd_pcm_access_t pcm_access[] =
    { SND_PCM_ACCESS_MMAP_INTERLEAVED, SND_PCM_ACCESS_RW_INTERLEAVED };

    mSetupPCM = false;

    if (pcm != NULL)
    {
        LOG_ERROR((aauto_audio, "%s, Failed to setup %s as pcm is already present", streamName, pcmname));
        return -EINVAL;
    }

    if (dump_config == true)
    {
        err = snd_output_stdio_attach(&output, stdout, 0);
        if (err < 0)
        {
            LOG_WARN((aauto_audio, "%s, Attaching stdout to snd_output failed", streamName));
            err = 0;
            output = NULL;
        }
    }

    snd_pcm_hw_params_alloca(&hw_params);
    snd_pcm_sw_params_alloca(&sw_params);
    LOGD_DEBUG((aauto_audio, "%s, CFG pcmname %s rate %u, channels %u, dir %s, format %s, " \
            "ideal period ms %u, buffer ideal periods %d, prefill ms %d", streamName, pcmname,
            rate, channels, snd_pcm_stream_name (dir), snd_pcm_format_name (format), ideal_period_ms, buffer_ideal_periods, prefill_ms));

    err = snd_pcm_open(&pcm, pcmname, dir, SND_PCM_NONBLOCK);
    LogSndStdErrorAndReturn(err, "snd_pcm_open", pcmname, output);

    err = snd_pcm_hw_params_any(pcm, hw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_any", pcmname, output);

    err = snd_pcm_hw_params_set_format(pcm, hw_params, format);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_format", pcmname, output);

    chmin = channels;
    chmax = channels;
    err = snd_pcm_hw_params_set_channels_minmax(pcm, hw_params, &chmin, &chmax);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_channels_min", pcmname, output);
    if ((chmin != channels) || (chmax != channels))
    {
        LOG_ERROR((aauto_audio, "Failed to set min max channels to %u (got %u to %u)", channels, chmin,
                chmax));
        return -EINVAL;
    }

    err = snd_pcm_hw_params_set_rate(pcm, hw_params, rate, 0);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_rate", pcmname, output);

    for (i = 0; i < (sizeof(pcm_access) / sizeof(pcm_access[0])); i++)
    {
        err = snd_pcm_hw_params_set_access(pcm, hw_params, pcm_access[i]);
        if (err == 0)
        {
            access = pcm_access[i];
            break;
        }
    }
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_access", pcmname, output);

    ideal_period_size = (ideal_period_ms * rate) / 1000;

    if (ideal_period_size == 0) {
        LOG_ERROR((aauto_audio, "%s, Invalid ideal period size", streamName));
        return -EINVAL;
    }

    silence_prefill = 0;
    if (dir == SND_PCM_STREAM_PLAYBACK) {
        prefill_ms_local = prefill_ms;

        /* using default */
        if (prefill_ms_local < 0) {
            LOGD_DEBUG((aauto_audio, "%s, Using default %d periods for silence prefill for %s.",
                        streamName, SILENCE_PREFILL_DEFAULT, pcmname));
            prefill_ms_local = SILENCE_PREFILL_DEFAULT * ideal_period_ms;
        }

        /* if silence is a multiple of ideal period ms use a different calc to avoid rounding issues */
        if (safe_modulo((unsigned int)prefill_ms_local, ideal_period_ms) == 0) {
            silence_prefill = ideal_period_size * ((unsigned int)prefill_ms_local / ideal_period_ms);
        } else {
            silence_prefill = (prefill_ms_local * rate) / 1000;
        }
    } else {
        if (prefill_ms > 0)
            LOG_WARN((aauto_audio, "%s, Silence prefill is only valid for playback - setting will be ignored for %s",
                    streamName, pcmname));
    }
    LOGD_DEBUG((aauto_audio, "%s, CFG %s ideal period size %lu, silence prefill frames %lu",
                streamName, pcmname, ideal_period_size, silence_prefill));

    err = -EINVAL;
    for (i = 0; (dividers[i] != 0) && err; i++)
    {
        mydiv = dividers[i];
        period_size = ideal_period_size;
        do
        {
            snd_pcm_uframes_t psize_min = period_size;
            snd_pcm_uframes_t psize_max = period_size;
            int mindir = -1;
            int maxdir = +1;
            err = snd_pcm_hw_params_set_period_size_minmax(pcm, hw_params, &psize_min, &mindir,
                    &psize_max, &maxdir);
            period_size = ideal_period_size / mydiv;
            mydiv *= dividers[i];
        } while ((period_size >= 16) && err);/*reasonable limit*/
    }
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_period_size", pcmname, output);

    buffer_periods_local = buffer_ideal_periods;
    /* using default */
    if (buffer_periods_local < 0) {
        LOGD_DEBUG((aauto_audio, "%s, Using default %d periods for buffer for %s.",
                streamName, BUFFER_PERIODS_DEFAULT, pcmname));
        buffer_periods_local = BUFFER_PERIODS_DEFAULT;
    }

    if (buffer_periods_local < 2) {
        LOG_ERROR((aauto_audio, "%s, Invalid buffer configuration for %s. Buffer has to be at least two periods",
                streamName, pcmname));
        return -EINVAL;
    }

    buffer_size = buffer_periods_local * ideal_period_size;
    /*extra space for larger silence prefill*/
    if (silence_prefill > (buffer_size - ideal_period_size)) {
        unsigned int silence_periods = ((silence_prefill - 1) / ideal_period_size) + 1;
        buffer_periods_local = silence_periods + 1;
        buffer_size = buffer_periods_local * ideal_period_size;
        LOGD_DEBUG((aauto_audio, "%s, Extending buffer for %s to %d periods.",
                streamName, pcmname, buffer_periods_local));
    }

    err = snd_pcm_hw_params_set_buffer_size_min(pcm, hw_params, &buffer_size);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_buffer_size_min", pcmname, output);

    err = snd_pcm_hw_params_set_buffer_size_near(pcm, hw_params, &buffer_size);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_set_buffer_size_near", pcmname, output);

    err = snd_pcm_hw_params(pcm, hw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params", pcmname, output);

    err = snd_pcm_hw_params_get_period_size(hw_params, &period_size, NULL);
    LogSndStdErrorAndReturn(err, "snd_pcm_hw_params_get_period_size", pcmname, output);

    LOGD_DEBUG((aauto_audio, "%s, CFG %s period size %lu, buffer size %lu",
            streamName, pcmname, period_size, buffer_size));

    err = snd_pcm_sw_params_current(pcm, sw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params_current", pcmname, output);

    if (dir == SND_PCM_STREAM_CAPTURE)
        err = snd_pcm_sw_params_set_start_threshold(pcm, sw_params, 1);
    else
        err = snd_pcm_sw_params_set_start_threshold(pcm, sw_params,
                ideal_period_size + silence_prefill);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params_set_start_threshold", pcmname, output);

    err = snd_pcm_sw_params_set_avail_min(pcm, sw_params, ideal_period_size);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params_set_avail_min", pcmname, output);

    err = snd_pcm_sw_params(pcm, sw_params);
    LogSndStdErrorAndReturn(err, "snd_pcm_sw_params", pcmname, output);

    bytes_per_frame = snd_pcm_format_size(format, channels);

    LOGD_DEBUG((aauto_audio, "%s, CFG %s complete", streamName, pcmname));

    DumpHwSwParams("configuration complete", pcmname, output);

    if (output != NULL)
    {
        snd_output_printf(output, "Dumping pcm configuration for %s", pcmname);\
        snd_pcm_dump(pcm, output);
        snd_output_flush(output);
        snd_output_close(output);
    }

    ReleaseBuffers();

    err = CreateBuffers();
    if (err < 0)
        return err;

    setAlsaAudioState(AlsaAudioStart);
    mStreamStarted = false;
    mStreamStopped = false;
    mSetupPCM = true;
    mStreamBroken = false;
    wait_done = false;
    startup_xfer = 0;

    LOGD_DEBUG((aauto_audio, "%s, SetupPCM() done. setup=%d, mStreamStarted=%d, mStreamBroken=%d, wait_done=%d",
                streamName, mSetupPCM, mStreamStarted, mStreamBroken, wait_done));

    return 0;
}

void AlsaAudioCommon::ReleasePCM()
{
    if (pcm != NULL)
    {
        LOGD_DEBUG((aauto_audio, "%s, ReleasePCM %s", streamName, snd_pcm_name(pcm)));

        if (mStreamStarted == true)
            snd_pcm_drop(pcm);

        snd_pcm_close(pcm);
    }
    pcm = NULL;

    mSetupPCM = false;

    ReleaseBuffers();
}

int AlsaAudioCommon::CreateBuffers()
{
    unsigned int silence_buffer_size = ideal_period_size * channels;

    if (dir != SND_PCM_STREAM_PLAYBACK)
        return 0;

    silence_buffer = malloc(snd_pcm_format_size(format, silence_buffer_size));

    if (silence_buffer == NULL)
    {
        LOG_ERROR((aauto_audio, "%s, AlsaAudioCommon failed to create buffers", streamName));

        ReleaseBuffers();

        return -ENOMEM;
    }

    snd_pcm_format_set_silence(format, silence_buffer, silence_buffer_size);

    return 0;
}

void AlsaAudioCommon::ReleaseBuffers()
{
    if (silence_buffer != NULL)
        free(silence_buffer);

    silence_buffer = NULL;
}

int AlsaAudioCommon::WriteSilence(const snd_pcm_uframes_t silence)
{
    int err = 0;
    snd_pcm_uframes_t silence_remain = silence;
    if (verbose) {
        LOGD_VERBOSE((aauto_audio, "%s, Prefill silence for %s silence_buffer=%p, silence=%ld",
                streamName, snd_pcm_name(pcm), silence_buffer, silence));
    }
    while ((silence_remain > 0) && (err == 0))
    {
        snd_pcm_uframes_t chunk = min(silence_remain, period_size);

        err = RwBuffer(silence_buffer, chunk);

        silence_remain -= chunk;
        if (err < 0)
        {
            LOG_ERROR((aauto_audio, "%s, RW SILENCE ERROR %d (%s) on dev %s", streamName, err, snd_strerror(err),
                    snd_pcm_name(pcm)));
            break;
        }
    }

    return err;
}

int AlsaAudioCommon::WaitPCM()
{
    int err = 0;
    int time_wait = 0;
    int tout = (wait_done==true)?(ideal_period_ms + 20):init_tout;

    if (true != wait_done) {
        LOGD_DEBUG((aauto_audio, "%s, WaitPCM() wait_done=false, pcm_wait=%d ms", streamName, init_tout));
    }

    /* wait in chunks until:
     * - timeout value (tout) is reached
     * - or the PCM to become ready (snd_pcm_wait() return 1)
     */
    for (time_wait=0; (0 == err) && (time_wait<tout); time_wait+=(ideal_period_ms + 20))
    {
        /* check if we should stop processing */
        if (getAlsaAudioState() != AlsaAudioStop) {
            /* Wait for a PCM to become ready */
            err = snd_pcm_wait(pcm, (ideal_period_ms + 20));
        } else {
            LOGD_DEBUG((aauto_audio, "%s, WaitPCM() mStreamState=%d. leave WaitPCM()", streamName, getAlsaAudioState()));
            err = 1;
            break; /* leave for loop */
        }
    }
    if (0 > err) { /* error occurred */
        LOG_ERROR((aauto_audio, "%s, WAIT ERR %d (%s) on dev %s, err=%d",
                   streamName, err, snd_strerror(err), snd_pcm_name(pcm), err));
    } else if (0 == err) { /* timeout occurred */
        err = -ENODATA;
        LOG_ERROR((aauto_audio, "%s, WaitPCM() WAIT TOUT on dev %s, err=%d",
                   streamName, snd_pcm_name(pcm), err));
    } else { /* 1 - PCM stream is ready for I/O */
        err = 0;
    }

    if (verbose) {
        LOGD_VERBOSE((aauto_audio, "%s, WaitPCM() mStreamState=%d, tout=%d, period_tout=%d, time_wait=%d, err=%d",
                      streamName, getAlsaAudioState(), tout, (ideal_period_ms + 20), time_wait, err ));
    }
    return err;
}

int AlsaAudioCommon::RwBuffer(void *buffer, snd_pcm_uframes_t xfer_size_)
{
    unsigned int retry_done = 0;
    int err = 0;
    char *buf_ptr;
    snd_pcm_uframes_t xfer_size;

    if (verbose)
    {
        LOGD_VERBOSE((aauto_audio, "%s, RwBuffer() buffer %p, size %lu",
                streamName, buffer, xfer_size_));
    }

    buf_ptr = (char*) buffer;
    xfer_size = xfer_size_;

    // TODO: Possible endless loop
    //       if WaitPCM() return 0, but xfer_size will be not decremented due to error.
    while ((xfer_size_ > 0) && (getAlsaAudioState() != AlsaAudioStop))
    {
        xfer_size = min(buffer_size, xfer_size_);
        xfer_size_ -= xfer_size;
        while ((xfer_size > 0) && (err == 0) && (getAlsaAudioState() != AlsaAudioStop))
        {
            do
            {/*EINTR should not occur, as we operate in nonblocking mode*/
                switch (access)
                {
                case SND_PCM_ACCESS_MMAP_INTERLEAVED:
                    if (dir == SND_PCM_STREAM_PLAYBACK)
                    {
                        err = snd_pcm_mmap_writei(pcm, buf_ptr, xfer_size);
                    }
                    else
                    {
                        err = snd_pcm_mmap_readi(pcm, buf_ptr, xfer_size);
                    }
                    break;
                case SND_PCM_ACCESS_RW_INTERLEAVED:
                    if (dir == SND_PCM_STREAM_PLAYBACK)
                    {
                        err = snd_pcm_writei(pcm, buf_ptr, xfer_size);
                    }
                    else
                    {
                        err = snd_pcm_readi(pcm, buf_ptr, xfer_size);
                    }
                    break;

                default:
                    LOG_ERROR((aauto_audio, "%s, RwBuffer() invalid access %d", streamName, access));
                    err = -EINVAL;
                    break;
                }
            } while (err == -EINTR);

            if (err >= 0)
            {
                if (err > (int) xfer_size)
                {
                    LOG_ERROR((aauto_audio, "%s, RwBuffer() invalid size %d", streamName, err));
                    err = -EINVAL;
                }
                else
                {
                    if ((err > 0) && (false == wait_done)) {
                        startup_xfer += err;
                        if (dir == SND_PCM_STREAM_CAPTURE) {
                            wait_done = true;
                        } else {
                            if (startup_xfer > buffer_size) {
                                wait_done = true;
                            }
                        }
                    }
                    buf_ptr += err * bytes_per_frame;
                    xfer_size -= err;

                    if (verbose)
                    {
                        LOGD_VERBOSE((aauto_audio, "%s, RwBuffer() xfer_size=%ld, xfered=%d",
                                streamName, xfer_size, err));
                    }
                    err = 0;
                }
            }

            if (err == -EAGAIN)
            {
                /*
                 Non period aligned write is buggy in libasound < 1.0.27:
                 Writing smaller buffers than period size can result in 100% CPU load, as snd_pcm_wait()
                 succeeds, while following write access returs -EAGAIN.
                 Fixed with commit 'PCM: Avoid busy loop in snd_pcm_write_areas() with rate plugin'.
                 As a workaround we do a short sleep if we get repeated EAGAIN's
                 */
                if (retry_done != 0)
                    usleep(1000);
                retry_done = 1;
                err = WaitPCM();

                if (verbose)
                {
                    LOGD_VERBOSE((aauto_audio, "%s, RwBuffer() snd_pcm_read/write(xfer_size=%ld) return EAGAIN and WaitPCM() with err=%d",
                            streamName, xfer_size, err));
                }
            }
        } // while ((xfer_size > 0) && (err == 0))
        if (0 > err) {
            LOG_ERROR((aauto_audio, "%s, RwBuffer() mStreamState=%d, err=%d",
                    streamName, getAlsaAudioState(), err));
            break;
        }
    } // while (xfer_size_ > 0)

    return err;

}

int AlsaAudioCommon::Recover()
{
    int err = 0;

    LOGD_DEBUG((aauto_audio, "%s, Recovering %s", streamName, snd_pcm_name(pcm)));

    mStreamStarted = false;
    err = snd_pcm_drop(pcm);
    if (err < 0)
    {
        return err;
    }
    err = snd_pcm_prepare(pcm);

    return err;
}

int AlsaAudioCommon::ReadWrite(void *buffer, size_t len)
{
    int err = 0;
    int tmp_err;
    int retry_cnt = 0;
    snd_pcm_uframes_t buffer_frames;

#define NUM_TRY_RECOVER 10

    if (mSetupPCM == false)
    {
        LOG_ERROR((aauto_audio, "%s, ReadWrite() must not be called before Setup is completed", streamName));
        return -EINVAL;
    }
    if (mStreamBroken != false)
    {
        LOG_ERROR((aauto_audio, "%s, ReadWrite() Stream is broken. In order to retry close and call Setup again", streamName));
        return -EINVAL;
    }

    // The "len" has to cast for x64 platform.
    if (safe_modulo((unsigned int)len, bytes_per_frame) != 0)
    {
        LOG_ERROR((aauto_audio, "%s, Buffer size %zu is not aligned to the bytes per frame %u given by " \
                "format %s and channels %u", streamName, len, bytes_per_frame, snd_pcm_format_name(format),
                channels));
        return -EINVAL;
    }

    buffer_frames = len / bytes_per_frame;

    if (verbose)
    {
        LOGD_VERBOSE((aauto_audio, "%s, New buffer : addr %p, len=%zu, bytes_per_frame=%u, frames %lu",
                streamName, buffer, len, bytes_per_frame, buffer_frames));
    }

    do
    {
        if ((mStreamStarted == false) && (dir == SND_PCM_STREAM_CAPTURE))
        {
            mStreamStarted = true;
            err = snd_pcm_start(pcm);
        }

        if ((mStreamStarted == false) && (dir == SND_PCM_STREAM_PLAYBACK))
        {
            /*side->started=1;*/
            err = WriteSilence(silence_prefill);
        }
        if (err < 0)
            break;/*no recover */

        err = WaitPCM();
        if (err == 0) {
            err = RwBuffer(buffer, buffer_frames);
            if (verbose) {
                LOGD_VERBOSE((aauto_audio, "%s, RwBuffer(Buffer addr %p, buffer_frames=%ld) = %d",
                                streamName, buffer, buffer_frames, err));
            }
        }
        if ((err == -EPIPE) || (err == -ENODATA))
        {
            tmp_err = Recover();    // set mStreamStarted to false
            if (recover_delay_ms != 0)
            {
                if (recover_delay_ms > 0)
                    usleep(recover_delay_ms * 1000);
                if (recover_delay_ms < 0)
                    break;
            }
            if (tmp_err < 0)
            {
                LOG_ERROR((aauto_audio, "%s, RECOVER ERR %d (%s)",
                        streamName, tmp_err, snd_strerror(tmp_err)));
                err = tmp_err;
            }
            else
            {
                if (dir == SND_PCM_STREAM_PLAYBACK)
                {
                    /* playback prefill and start on next write call*/
                    err = 0;
                    break; // jump out of while()-loop
                }
                /* In case of SND_PCM_STREAM_CAPTURE,
                 * we try within the while()-loop to recover. */
            }
        }

        if (err == 0)
        {
            /* playback direction should already be started automatically */
            if ((false == mStreamStarted) && (dir == SND_PCM_STREAM_PLAYBACK))
            {
                mStreamStarted = true;
//                snd_pcm_start(pcm);
            }
        }

        retry_cnt++;
    } while (((err == -EPIPE) || (err == -ENODATA)) && (retry_cnt <= NUM_TRY_RECOVER) && (getAlsaAudioState() < AlsaAudioStop));

    if (err < 0)
    {
        mStreamBroken = true;
        LOG_ERROR((aauto_audio, "%s, ReadWrite() retry_cnt=%d, err=%d, mStreamStarted=%d, mStreamBroken=%d => StopStreaming",
                streamName, retry_cnt, err, mStreamStarted, mStreamBroken));
        StopStreaming();
    }
    return err;
}

int AlsaAudioCommon::StopStreaming()
{
    int err = 0;
    bool wasStopped = mStreamStopped;

    /* calling snd_pcm_drain() or snd_pcm_drop() twice could cause
     * unexpected behavior in the underlying audio plug-in like ALSA */
    if (true == wasStopped) {
        LOG_WARN((aauto_audio, "%s, StopStreaming() was already called", streamName));
        return err;
    }
    /* mStreamStopped set to false in SetupPCM()
     * which must be called as preparation to start streaming */
    mStreamStopped = true;

    LOGD_DEBUG((aauto_audio, "%s, StopStreaming() mStreamStarted=%d, mStreamBroken=%d", streamName, mStreamStarted, mStreamBroken));

    if (pcm != NULL) {
        /* snd_pcm_drain() shall be done only if there was no error */
        if ((mStreamBroken != true) && (dir == SND_PCM_STREAM_PLAYBACK)) {
            /* ensure that silence is filled up period aligned */
            if (safe_modulo(startup_xfer, period_size)) {
                WriteSilence(period_size - (safe_modulo(startup_xfer, period_size)));
            }

            /* set nonblock mode to block */
            err = snd_pcm_nonblock(pcm, 0);
            if (0 == err) {
                /* wait for all pending frames to be played and then stop the PCM */
                err = snd_pcm_drain(pcm);
                if (0 != err) {
                    LOG_ERROR((aauto_audio, "%s, Draining %s failed with %d(%s)",
                            streamName, snd_pcm_name(pcm), err, snd_strerror(err)));
                    mStreamBroken = true;
                }
                /* set nonblock mode to nonblock */
                snd_pcm_nonblock(pcm, 1);
            } else {
                LOG_ERROR((aauto_audio, "%s, Switching %s to blocking mode failed with %d(%s)",
                        streamName, snd_pcm_name(pcm), err, snd_strerror(err)));
                mStreamBroken = true;
            }
        }
        if (0 == err) {
            /* This function stops the PCM immediately.
             * The pending samples on the buffer are ignored */
            err = snd_pcm_drop(pcm);
            if (err != 0) {
                LOG_ERROR((aauto_audio, "%s, Dropping %s failed with %d(%s)",
                        streamName, snd_pcm_name(pcm), err, snd_strerror(err)));
                mStreamBroken = true;
            }
        }
    } else {
        LOG_ERROR((aauto_audio, "%s, StopStreaming() PCM is not opened", streamName));
        err = -EINVAL;
    }

    mStreamStarted = false;
    startup_xfer = 0;
    /* mSetupPCM set only in SetupPCM() and ReleasePCM() */
    wait_done = false;

    return err;
}

int AlsaAudioCommon::Delay(snd_pcm_sframes_t *delay)
{
    int err;

    if (pcm == NULL)
    {
        LOG_ERROR((aauto_audio, "%s, Failed to get delay as PCM is not opened", streamName));
        return -EINVAL;
    }

    err = snd_pcm_delay(pcm, delay);
    if (err < 0)
    {
        LOG_ERROR((aauto_audio, "%s, Delay function of %s failed with %d(%s)",
                streamName, snd_pcm_name(pcm), err, snd_strerror(err)));
        return err;
    }
    if (*delay < 0)
    {
        LOG_WARN((aauto_audio, "%s, Delay of %s is negative %ld", streamName, snd_pcm_name(pcm), *delay));
    }
    if (verbose) {
        LOGD_VERBOSE((aauto_audio, "%s, Delay of %s is %ld", streamName, snd_pcm_name(pcm), *delay));
    }

    return err;
}

}
} // namespace adit { namespace aauto
